<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Widget_Image_Litae extends Widget_Base {

	public function get_name() {
		return 'image-litae';
	}

	public function get_title() {
		return esc_html__( 'Image Litae', 'elementor-custom-element'  );
	}

	public function get_icon() {
		return 'eicon-insert-image';
	}

        public function get_categories() {
            return ['litae-elements'];
        }
        
	protected function _register_controls() {
		$this->start_controls_section(
			'section_image',
			[
				'label' => esc_html__( 'Image', 'elementor-custom-element'  ),
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__( 'Choose Image', 'elementor-custom-element'  ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name' => 'image', // Actually its `image_size`
				'label' => esc_html__( 'Image Size', 'elementor-custom-element'  ),
				'default' => 'large',
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => esc_html__( 'Alignment', 'elementor-custom-element'  ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
				],
			]
		);
                $this->add_control(
                'is_effect',
                    [
                        'label' => esc_html( 'Reveal effect', 'elementor-custom-element' ),
                        'type' => Controls_Manager::SWITCHER,
                        'default' => false,
                    ]
                );
                $this->add_control(
                    'animation_color',
                    [
                        'label' => __( 'Animation Color', 'elementor-custom-element' ),
                        'type' => Controls_Manager::COLOR,
                        'scheme' => [
                            'type' => Scheme_Color::get_type(),
                            'value' => Scheme_Color::COLOR_1,
                        ],
                        'condition' => [
                            'is_effect' => 'on'
                        ]
                    ]
                );
                $this->add_control(
                    'animation_direction',
                        [
                           'label' => esc_html__( 'Animation direction', 'elementor-custom-element' ),
                           'type' => Controls_Manager::SELECT,
                           'label_block' => true,
                           'options' => [
                              'lr' => esc_html__( 'Left to right', 'elementor-custom-element' ),
                              'rl' => esc_html__( 'Right to left', 'elementor-custom-element' ),
                              'tb' => esc_html__( 'Top to bottom', 'elementor-custom-element' ),
                              'bt' => esc_html__( 'Bottom to top', 'elementor-custom-element' ),
                           ],
                           'default' => 'lr',
                           'multiple' => false,
                           'condition' => [
                               'is_effect' => 'on'
                           ]
                        ]
                );
                $this->add_control(
                    'animation_delay',
                    [
                        'label' => __( 'Animation delay', 'elementor-custom-element' ),
                        'type' => Controls_Manager::SLIDER,
                        'default' => [
                            'size' => 250,
                        ],
                        'range' => [
                            'px' => [
                                'min' => 0,
                                'max' => 2000,
                                'step' => 10,
                            ],
                        ],
                        'size_units' => [ 'px' ],
                        'condition' => [
                            'is_effect' => 'on'
                        ],
                        'separator' => ['after']
                    ]
                );
		$this->add_control(
			'caption',
			[
				'label' => esc_html__( 'Caption', 'elementor-custom-element'  ),
				'type' => Controls_Manager::TEXT,
				'default' => '',
				'placeholder' => esc_html__( 'Enter your caption about the image', 'elementor-custom-element'  ),
				'title' => esc_html__( 'Input image caption here', 'elementor-custom-element'  ),
			]
		);

		$this->add_control(
			'link_to',
			[
				'label' => esc_html__( 'Link to', 'elementor-custom-element'  ),
				'type' => Controls_Manager::SELECT,
				'default' => 'none',
				'options' => [
					'none' => esc_html__( 'None', 'elementor-custom-element'  ),
					'file' => esc_html__( 'Media File', 'elementor-custom-element'  ),
					'custom' => esc_html__( 'Custom URL', 'elementor-custom-element'  ),
				],
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link to', 'elementor-custom-element'  ),
				'type' => Controls_Manager::URL,
				'placeholder' => esc_html__( 'http://your-link.com', 'elementor-custom-element'  ),
				'condition' => [
					'link_to' => 'custom',
				],
				'show_label' => false,
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'elementor-custom-element'  ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image',
			[
				'label' => esc_html__( 'Image', 'elementor-custom-element'  ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'space',
			[
				'label' => esc_html__( 'Size (%)', 'elementor-custom-element'  ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 100,
					'unit' => '%',
				],
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-image img' => 'max-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'opacity',
			[
				'label' => esc_html__( 'Opacity (%)', 'elementor-custom-element'  ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 1,
				],
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-image img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_control(
			'hover_animation',
			[
				'label' => esc_html__( 'Hover Animation', 'elementor-custom-element'  ),
				'type' => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'image_border',
				'label' => esc_html__( 'Image Border', 'elementor-custom-element'  ),
				'selector' => '{{WRAPPER}} .elementor-image img',
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'elementor-custom-element'  ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .elementor-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'image_box_shadow',
				'selector' => '{{WRAPPER}} .elementor-image img',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_caption',
			[
				'label' => esc_html__( 'Caption', 'elementor-custom-element'  ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'caption_align',
			[
				'label' => esc_html__( 'Alignment', 'elementor-custom-element'  ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'elementor-custom-element'  ),
						'icon' => 'fa fa-align-justify',
					],
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .widget-image-caption' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label' => esc_html__( 'Text Color', 'elementor-custom-element'  ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .widget-image-caption' => 'color: {{VALUE}};',
				],
				'scheme' => [
					'type' => Scheme_Color::get_type(),
					'value' => Scheme_Color::COLOR_3,
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'caption_typography',
				'selector' => '{{WRAPPER}} .widget-image-caption',
				'scheme' => Scheme_Typography::TYPOGRAPHY_3,
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings();

		if ( empty( $settings['image']['url'] ) ) {
			return;
		}
                
                $watcher = ( $settings['is_effect']!='on' )? '' : 'watcher';

                $animation_settings = [
                    'bgcolor'     => esc_attr( $settings['animation_color'] ),
                    'direction'   => esc_attr( $settings['animation_direction'] ),
                    'delay'       => absint( $settings['animation_delay']['size'] ),
                ];
                
                $data_rev_settings = esc_attr( wp_json_encode( $animation_settings ) );

		$has_caption = ! empty( $settings['caption'] );

		$this->add_render_attribute( 'wrapper', 'class', 'elementor-image' );

		if ( ! empty( $settings['shape'] ) ) {
			$this->add_render_attribute( 'wrapper', 'class', 'elementor-image-shape-' . $settings['shape'] );
		}
                
                if( ! ''==$watcher){
                    $this->add_render_attribute('wrapper', 'class', $watcher);
                    $this->add_render_attribute('wrapper', 'data-rev-settings', $data_rev_settings);
                }
                
		$link = $this->get_link_url( $settings );

		if ( $link ) {
			$this->add_render_attribute( 'link', 'href', $link['url'] );

			if ( ! empty( $link['is_external'] ) ) {
				$this->add_render_attribute( 'link', 'target', '_blank' );
			}
		} ?>
		<div <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
		<?php
		if ( $has_caption ) : ?>
			<figure class="wp-caption">
		<?php endif;

		if ( $link ) : ?>
				<a <?php echo $this->get_render_attribute_string( 'link' ); ?>>
		<?php endif;

		echo Group_Control_Image_Size::get_attachment_image_html( $settings );

		if ( $link ) : ?>
				</a>
		<?php endif;

		if ( $has_caption ) : ?>
				<figcaption class="widget-image-caption wp-caption-text"><?php echo $settings['caption']; ?></figcaption>
		<?php endif;

		if ( $has_caption ) : ?>
			</figure>
		<?php endif; ?>
		</div>
		<?php
	}

	protected function _content_template() {
		?>
		<# if ( '' !== settings.image.url ) {
			var image = {
				id: settings.image.id,
				url: settings.image.url,
				size: settings.image_size,
				dimension: settings.image_custom_dimension,
				model: editModel
			};

			var image_url = elementor.imagesManager.getImageUrl( image );

			if ( ! image_url ) {
				return;
			}

			var link_url;

			if ( 'custom' === settings.link_to ) {
				link_url = settings.link.url;
			}

			if ( 'file' === settings.link_to ) {
				link_url = settings.image.url;
			}

			#><div class="elementor-image{{ settings.shape ? ' elementor-image-shape-' + settings.shape : '' }}"><#
			var imgClass = '',
				hasCaption = '' !== settings.caption;

			if ( '' !== settings.hover_animation ) {
				imgClass = 'elementor-animation-' + settings.hover_animation;
			}

			if ( hasCaption ) {
				#><figure class="wp-caption"><#
			}

			if ( link_url ) {
					#><a href="{{ link_url }}"><#
			}
						#><img src="{{ image_url }}" class="{{ imgClass }}" /><#

			if ( link_url ) {
					#></a><#
			}

			if ( hasCaption ) {
					#><figcaption class="widget-image-caption wp-caption-text">{{{ settings.caption }}}</figcaption><#
			}

			if ( hasCaption ) {
				#></figure><#
			}

			#></div><#
		} #>
		<?php
	}

	private function get_link_url( $instance ) {
		if ( 'none' === $instance['link_to'] ) {
			return false;
		}

		if ( 'custom' === $instance['link_to'] ) {
			if ( empty( $instance['link']['url'] ) ) {
				return false;
			}
			return $instance['link'];
		}

		return [
			'url' => $instance['image']['url'],
		];
	}
}
