<?php

namespace Elementor;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

class Widget_Intro_Slider extends Widget_Base {

    public function get_name() {
        return 'litae-intro-slider';
    }

    public function get_title() {
        return esc_html__('Intro Slider', 'elementor-custom-element');
    }

    public function get_icon() {
        return 'fa fa-sliders';
    }

    public function get_categories() {
        return ['litae-elements'];
    }

    public function get_script_depends() {
        return ['jquery-slick'];
    }

    protected function _register_controls() {
        $this->start_controls_section(
            'section_intro_content',
            [
                'label' => esc_html__( 'Intro slider', 'elementor-custom-element' ),
            ]
        );
        $this->add_control(
            'item_height',
            [
                'label' => esc_html__( 'Item height', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => '100vh',
                'options' => [
                    '50vh'  => esc_html__( '50vh', 'elementor-custom-element' ),
                    '80vh'  => esc_html__( '80vh', 'elementor-custom-element' ),
                    '100vh' => esc_html__( '100vh', 'elementor-custom-element' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .intro-slider .item' => 'height: {{VALUE}};',
                ],
            ]
        );
        /*
         * Repeater
         */
        $this->add_control(
            'slides',
            [
                'label' => esc_html__( 'Items', 'elementor-custom-element' ),
                'type' => Controls_Manager::REPEATER,
                'default' => [],
                'title' => 'Custom item',
                'fields' => [
                    [
                        'name' => 'image',
                        'label' => esc_html__( 'BG Image', 'elementor-custom-element' ),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                                'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'name' => 'text',
                        'label' => esc_html__( 'Text', 'elementor-custom-element' ),
                        'type' => Controls_Manager::TEXTAREA,
                        'label_block' => true,
                        'placeholder' => esc_html__( 'Say something', 'elementor-custom-element' ),
                    ],
                ],
                'title_field' => '{{{ text }}}',
            ]
        );
        /*
         * end Repeater
         */
        /*
         * Scroll arrow
         */
        $this->add_control(
            'scroll_link',
            [
                'label' => esc_html__( 'Scroll arrow', 'elementor-custom-element' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => '',
                'label_on' => esc_html__( 'Show', 'elementor-custom-element' ),
                'label_off' => esc_html__( 'Hide', 'elementor-custom-element' ),
                
            ]
        );
        $this->add_responsive_control(
            'scroll_link_position',
                [
                    'label' => esc_html__( 'Arrow position', 'elementor-custom-element' ),
                    'type' => Controls_Manager::CHOOSE,
                    'options' => [
                        'left' => [
                            'title' => esc_html__( 'Left', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-left',
                        ],
                        'right' => [
                            'title' => esc_html__( 'Right', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-right',
                        ],
                    ],
                    'default' => 'left',
                    'selectors' => [
                        '{{WRAPPER}} .intro-wrapper .scroll-to-link a' => 'float: {{VALUE}};',
                    ],
                    'condition' => [
                        'scroll_link' => [ 'yes' ]
                    ]
                ]
        );
        $this->add_control(
            'scroll_section_id',
            [
                'label' => esc_html__( 'Section ID', 'elementor-custom-element' ),
                'description' => esc_html__( 'ID of section where you want to scroll', 'elementor-custom-element' ),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'scroll_link' => [ 'yes' ]
                ]
            ]
        );
        $this->add_control(
            'scroll_section_arrow_title',
            [
                'label' => esc_html__( 'Arrow title', 'elementor-custom-element' ),
                'default' => esc_html__( 'scroll down', 'elementor-custom-element' ),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'scroll_link' => [ 'yes' ]
                ]
            ]
        );
        /*
         * end Scroll arrow
         */
        $this->end_controls_section();
        /*
         * Slider Options
         */
        $this->start_controls_section(
            'section_slider_options',
            [
                'label' => esc_html__( 'Slider Options', 'elementor-custom-element' ),
            ]
        );      
        $this->add_control(
            'navigation',
            [
                'label' => esc_html__( 'Navigation', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'both',
                'options' => [
                    'both' => esc_html__( 'Arrows and Dots', 'elementor-custom-element' ),
                    'arrows' => esc_html__( 'Arrows', 'elementor-custom-element'),
                    'dots' => esc_html__( 'Dots', 'elementor-custom-element' ),
                    'none' => esc_html__( 'None', 'elementor-custom-element' ),
                ],
            ]
        );
                
        $this->add_responsive_control(
            'navigation_position',
                [
                    'label' => esc_html__( 'Navigation aligmnet', 'elementor-custom-element' ),
                    'type' => Controls_Manager::CHOOSE,
                    'options' => [
                        'left' => [
                            'title' => esc_html__( 'Left', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-left',
                        ],
                        'center' => [
                            'title' => esc_html__( 'Center', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-center',
                        ],
                        'right' => [
                            'title' => esc_html__( 'Right', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-right',
                        ],
                    ],
                    'default' => 'center',
                    'selectors' => [
                        '{{WRAPPER}} .intro-slider ul.slick-dots ' => 'text-align: {{VALUE}};',
                    ],
                    'condition' => [
                        'navigation' => [ 'dots', 'both' ]
                    ]
                ]
        );
                
        $this->add_control(
            'pause_on_hover',
            [
                'label' => esc_html__( 'Pause on Hover', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'yes',
                'options' => [
                    'yes' => esc_html__( 'Yes', 'elementor-custom-element' ),
                    'no' => esc_html__( 'No', 'elementor-custom-element' ),
                ],
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__( 'Autoplay', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'yes',
                'options' => [
                    'yes' => esc_html__( 'Yes', 'elementor-custom-element' ),
                    'no' => esc_html__( 'No', 'elementor-custom-element' ),
                ],
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label' => esc_html__( 'Autoplay Speed', 'elementor-custom-element' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 5000,
            ]
        );

        $this->add_control(
            'infinite',
            [
                'label' => esc_html__( 'Infinite Loop', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'yes',
                'options' => [
                    'yes' => esc_html__( 'Yes', 'elementor-custom-element' ),
                    'no' => esc_html__( 'No', 'elementor-custom-element' ),
                ],
            ]
        );

        $this->add_control(
            'effect',
            [
                'label' => esc_html__( 'Effect', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'slide',
                'options' => [
                    'slide' => esc_html__( 'Slide', 'elementor-custom-element' ),
                    'fade' => esc_html__( 'Fade', 'elementor-custom-element' ),
                ],
            ]
        );

        $this->add_control(
            'speed',
            [
                'label' => esc_html__( 'Animation Speed', 'elementor-custom-element' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 500,
            ]
        );

        $this->add_control(
            'direction',
            [
                'label' => esc_html__( 'Direction', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'ltr',
                'options' => [
                    'ltr' => esc_html__( 'Left', 'elementor-custom-element' ),
                    'rtl' => esc_html__( 'Right', 'elementor-custom-element' ),
                ],
            ]
        );

        $this->end_controls_section();
        /*
         * end Slider Options
         */
        /*
         * Style
         */
        $this->start_controls_section(
            'section_style_font',
            [
                'label' => esc_html__( 'Font', 'elementor-custom-element' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                        'navigation' => [ 'arrows', 'dots', 'both' ],
                ],
            ]
        );
        
        $this->add_control(
            'font_color',
            [
                'label' => esc_html__( 'Text Color', 'elementor-custom-element' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .intro-slider .home-text h1' => 'color: {{VALUE}};',
                ]
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'scheme' => Scheme_Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .intro-slider .home-text h1',
            ]
        );
        
        $this->end_controls_section();
        
        $this->start_controls_section(
            'section_style_navigation',
            [
                'label' => esc_html__( 'Navigation', 'elementor-custom-element' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                        'navigation' => [ 'arrows', 'dots', 'both' ],
                ],
            ]
        );


        $this->add_control(
            'heading_style_arrows',
            [
                'label' => esc_html__( 'Arrows', 'elementor-custom-element' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                        'navigation' => [ 'arrows', 'both' ],
                ],
            ]
        );

        $this->add_control(
            'arrows_position',
            [
                'label' => esc_html__( 'Arrows Position', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'inside',
                'options' => [
                        'inside' => esc_html__( 'Inside', 'elementor-custom-element' ),
                        'outside' => esc_html__( 'Outside', 'elementor-custom-element' ),
                ],
                'condition' => [
                        'navigation' => [ 'arrows', 'both' ],
                ],
            ]
        );

        $this->add_control(
            'arrows_size',
            [
                'label' => esc_html__( 'Arrows Size', 'elementor-custom-element' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 120,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .intro-slider .slick-next:before , {{WRAPPER}} .intro-slider .slick-prev:before ' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                        'navigation' => [ 'arrows', 'both' ],
                ],
            ]
        );


        $this->add_control(
            'arrows_color',
            [
                'label' => esc_html__( 'Arrows Color', 'elementor-custom-element' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .intro-slider .slick-next:before, {{WRAPPER}} .intro-slider .slick-prev:before' => 'color: {{VALUE}};',
                ],
                'condition' => [
                        'navigation' => [ 'arrows', 'both' ],
                ],
            ]
        );

        $this->add_control(
            'heading_style_dots',
            [
                'label' => esc_html__( 'Dots', 'elementor-custom-element' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                        'navigation' => [ 'dots', 'both' ],
                ],
            ]
        );

        $this->add_control(
            'dots_position',
            [
                'label' => esc_html__( 'Dots Position', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'outside',
                'options' => [
                        'outside' => esc_html__( 'Outside', 'elementor-custom-element' ),
                        'inside' => esc_html__( 'Inside', 'elementor-custom-element' ),
                ],
                'condition' => [
                        'navigation' => [ 'dots', 'both' ],
                ],
            ]
        );

        $this->add_control(
            'dots_size',
            [
                'label' => esc_html__( 'Dots Size', 'elementor-custom-element' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                        'px' => [
                                'min' => 10,
                                'max' => 16,
                        ],
                ],
                'selectors' => [
                        '{{WRAPPER}} .intro-slider ul.slick-dots li button:before' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                        'navigation' => [ 'dots', 'both' ],
                ],
            ]
        );

        $this->add_control(
            'dots_color',
            [
                'label' => esc_html__( 'Dots Color', 'elementor-custom-element' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                        '{{WRAPPER}} .intro-slider ul.slick-dots li button:before' => 'color: {{VALUE}};',
                ],
                'condition' => [
                        'navigation' => [ 'dots', 'both' ],
                ],
            ]
        );

		$this->end_controls_section();
        /*
         * end Style
         */
    }

    protected function render() {
        
        $settings = $this->get_settings();
        
        if( empty($settings['slides']) ) {
            return;
        }
        $is_rtl = ( 'rtl' === $settings['direction'] );
        $direction = $is_rtl ? 'rtl' : 'ltr';
        $show_dots = ( in_array( $settings['navigation'], [ 'dots', 'both' ] ) );
        $show_arrows = ( in_array( $settings['navigation'], [ 'arrows', 'both' ] ) );
        $is_scroll_link = ( 'yes' === $settings['scroll_link'] );
        $scroll_section_id = esc_attr( $settings['scroll_section_id'] );
        $scroll_arrow_title = esc_html( $settings['scroll_section_arrow_title'] );
        $scroll_arrow_html = '';
        if ( $is_scroll_link ):
            $scroll_arrow_html .= '<div class="scroll-to-link">';
            $scroll_arrow_html .= '<div class="container">';
            $scroll_arrow_html .= '<div class="twelve columns">';
            $scroll_arrow_html .= '<a href="#'.$scroll_section_id.'" class="tipped" data-title="'.$scroll_arrow_title.'" data-tipper-options="{"direction":"top","follow":"true","margin":35}"></a>';
            $scroll_arrow_html .= '</div>';
            $scroll_arrow_html .= '</div>';
            $scroll_arrow_html .= '</div>';
        endif;
        
        $slick_options = [
            'slidesToShow' => 1,//absint( $settings['slides_to_show'] ),
            'autoplaySpeed' => absint( $settings['autoplay_speed'] ),
            'autoplay' => ( 'yes' === $settings['autoplay'] ),
            'infinite' => ( 'yes' === $settings['infinite'] ),
            'pauseOnHover' => ( 'yes' === $settings['pause_on_hover'] ),
            'speed' => absint( $settings['speed'] ),
            'arrows' => $show_arrows,
            'dots' => $show_dots,
            'rtl' => $is_rtl,
        ];
        if ( 'fade' === $settings['effect'] ){
            $slick_options['fade'] = true;
        }
        
        $carousel_classes = [];

        if ( $show_arrows ) {
            $carousel_classes[] = 'slick-arrows-' . $settings['arrows_position'];
        }

        if ( $show_dots ) {
            $carousel_classes[] = 'slick-dots-' . $settings['dots_position'];
        }
        $carousel_classes_esc = esc_attr( implode( ' ', $carousel_classes ) );
        
        $data_slider_options = esc_attr( wp_json_encode( $slick_options ) );
        
        $slides_html = '';
        
        foreach( $settings['slides'] as $item ):
            if( '' == $item['image']['url'] ){
                continue;
            }
            //style="background-image:url('. esc_url($item['image']['url']).')"
            $slides_html .= '<div class="item" style="background-image:url('. esc_url($item['image']['url']).')">';
            $slides_html .= '<div class="home-text">';
            $slides_html .= '<div class="container">';
            $slides_html .= '<div class="twelve columns">';
            $slides_html .= '<h1>'. nl2br( $item['text'] ).'</h1>';
            $slides_html .= '</div>';
            $slides_html .= '</div>';
            $slides_html .= '</div>';
            $slides_html .= '</div>';
        endforeach;
        
        
        echo <<<HTML
<div class="intro-wrapper" dir="{$direction}">
<div class="intro-slider {$carousel_classes_esc}" data-slider_options="{$data_slider_options}">
    {$slides_html}
</div>
    {$scroll_arrow_html}
</div>
HTML;
    }

    protected function _content_template() {
        
    }


}
