<?php

namespace Elementor;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

class Widget_Text_Rotator extends Widget_Base {
    
    public function get_name() {
        return 'litae-text-rotator';
    }

    public function get_title() {
        return esc_html__('Text Rotator', 'elementor-custom-element');
    }

    public function get_icon() {
        return 'fa fa-superpowers';
    }

    public function get_categories() {
        return ['litae-elements'];
    }

    public function get_script_depends() {
        return ['jquery'];
    }
    
    protected function _register_controls() {
        $this->start_controls_section(
            'section_intro_content',
            [
                'label' => esc_html__( 'Text Rotator', 'elementor-custom-element' ),
            ]
        );
        $this->add_control(
            'item_height',
            [
                'label' => esc_html__( 'Box height', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => '100vh',
                'options' => [
                    '50vh'  => esc_html__( '50vh', 'elementor-custom-element' ),
                    '80vh'  => esc_html__( '80vh', 'elementor-custom-element' ),
                    '100vh' => esc_html__( '100vh', 'elementor-custom-element' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .text-rotator' => 'height: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'text',
            [
                'label' => esc_html__( 'Text', 'elementor-custom-element' ),
                'type' => Controls_Manager::TEXTAREA,
            ]
        );
        $this->add_control(
            'text_color',
            [
                'label' => esc_html__( 'Text Color', 'elementor-custom-element' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rotator-wrapp h1' => 'color: {{VALUE}};',
                ]
            ]
        );
        $this->add_control(
            'text_bg_color',
            [
                'label' => esc_html__( 'Selected text bg color', 'elementor-custom-element' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cd-words-wrapper.selected' => 'background-color: {{VALUE}};',
                ]
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'scheme' => Scheme_Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .rotator-wrapp h1',
            ]
        );
        $this->end_controls_section();
        
        /*
         * Rotator settings
         */
        $this->start_controls_section(
            'section_additional_options',
            [
                'label' => esc_html__( 'Rotator settings', 'elementor-custom-element' ),
            ]
        );
        $this->add_control(
            'type_letters_delay',
            [
                'label' => esc_html__( 'Type letters delay (ms)', 'elementor-custom-element'  ),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 50,
                    'unit' => 'ms'
                ],
                'range' => [
                    'ms' =>[
                        'min' => 20,
                        'max' => 150,
                    ],
                ],
            ]
        );
        $this->add_control(
            'section_duration',
            [
                'label' => esc_html__( 'Selected duration time (ms)', 'elementor-custom-element' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 2000,
                    'unit' => 'ms'
                ],
                'range' => [
                    'ms' =>[
                        'min' => 500,
                        'max' => 4000,
                    ],
                ],
            ]
        );
        $this->add_control(
            'is_effect',
                [
                    'label' => esc_html( 'Reveal effect', 'elementor-custom-element' ),
                    'type' => Controls_Manager::SWITCHER,
                    'default' => false,
                    'description' => 'see: Theme Options » General Settings » Colors » Reveal Settings to choose Primary and Secondary Color'
                ]
        );
        $this->add_control(
            'animation_color',
            [
                'label' => __( 'Animation Color', 'elementor-custom-element' ),
                'type' => Controls_Manager::COLOR,
                'scheme' => [
                    'type' => Scheme_Color::get_type(),
                    'value' => Scheme_Color::COLOR_1,
                ],
                'condition' => [
                    'is_effect' => 'on'
                ]
            ]
        );
        $this->add_control(
            'animation_direction',
                [
                   'label' => esc_html__( 'Animation direction', 'elementor-custom-element' ),
                   'type' => Controls_Manager::SELECT,
                   'label_block' => true,
                   'options' => [
                      'lr' => esc_html__( 'Left to right', 'elementor-custom-element' ),
                      'rl' => esc_html__( 'Right to left', 'elementor-custom-element' ),
                      'tb' => esc_html__( 'Top to bottom', 'elementor-custom-element' ),
                      'bt' => esc_html__( 'Bottom to top', 'elementor-custom-element' ),
                   ],
                   'default' => 'lr',
                   'multiple' => false,
                   'condition' => [
                       'is_effect' => 'on'
                   ]
                ]
        );
        $this->add_control(
            'animation_delay',
            [
                'label' => __( 'Animation delay', 'elementor-custom-element' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 250,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 2000,
                        'step' => 10,
                    ],
                ],
                'size_units' => [ 'px' ],
                'condition' => [
                    'is_effect' => 'on'
                ]
            ]
        );
        /*
         * Scroll arrow
         */
        $this->add_control(
            'scroll_link',
            [
                'label' => esc_html__( 'Scroll arrow', 'elementor-custom-element' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => '',
                'label_on' => esc_html__( 'Show', 'elementor-custom-element' ),
                'label_off' => esc_html__( 'Hide', 'elementor-custom-element' ),
                
            ]
        );
        $this->add_responsive_control(
            'scroll_link_position',
                [
                    'label' => esc_html__( 'Arrow position', 'elementor-custom-element' ),
                    'type' => Controls_Manager::CHOOSE,
                    'options' => [
                        'left' => [
                            'title' => esc_html__( 'Left', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-left',
                        ],
                        'right' => [
                            'title' => esc_html__( 'Right', 'elementor-custom-element' ),
                            'icon' => 'fa fa-align-right',
                        ],
                    ],
                    'default' => 'left',
                    'selectors' => [
                        '{{WRAPPER}} .intro-wrapper .scroll-to-link a' => 'float: {{VALUE}};',
                    ],
                    'condition' => [
                        'scroll_link' => [ 'yes' ]
                    ]
                ]
        );
        $this->add_control(
            'scroll_section_id',
            [
                'label' => esc_html__( 'Section ID', 'elementor-custom-element' ),
                'description' => esc_html__( 'ID of section where you want to scroll', 'elementor-custom-element' ),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'scroll_link' => [ 'yes' ]
                ]
            ]
        );
        $this->add_control(
            'scroll_section_arrow_title',
            [
                'label' => esc_html__( 'Arrow title', 'elementor-custom-element' ),
                'default' => esc_html__( 'scroll down', 'elementor-custom-element' ),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'scroll_link' => [ 'yes' ]
                ]
            ]
        );
        /*
         * end Scroll arrow
         */
        
        $this->end_controls_section();
        /*
         * end Rotator settings
         */
        
    }
    
    protected function render() {
        $settings = $this->get_settings();
        
        $text = $settings['text'];
        
        $is_effect = ( "on" === $settings['is_effect'] );
        
        $watcher = $is_effect ? 'watcher' : '';
        
        $animation_settings = [
            'bgcolor'     => $settings['animation_color'],
            'direction' => $settings['animation_direction'],
            'delay'     => absint( $settings['animation_delay']['size'] ),
        ];

        $data_rev_settings = esc_attr( wp_json_encode( $animation_settings ) );
        
        $is_scroll_link = ( 'yes' === $settings['scroll_link'] );
        $scroll_section_id = esc_attr( $settings['scroll_section_id'] );
        $scroll_arrow_title = esc_html( $settings['scroll_section_arrow_title'] );
        $scroll_arrow_html = '';
        if ( $is_scroll_link ):
            $scroll_arrow_html .= '<div class="scroll-to-link">';
            $scroll_arrow_html .= '<a href="#'.$scroll_section_id.'" class="tipped" data-title="'.$scroll_arrow_title.'" data-tipper-options="{"direction":"top","follow":"true","margin":35}"></a>';
            $scroll_arrow_html .= '</div>';
        endif;
        
        $rotator_options = [
            'rotatorTypeLettersDelay' => absint( $settings['type_letters_delay']['size'] ),
            'rotatorSelectedDuration' => absint( $settings['section_duration']['size'] ),
        ];
        
        $data_rotator_options = esc_attr( wp_json_encode( $rotator_options ) );
        
        echo <<<HTML
<div class="text-rotator" data-rotator_options="{$data_rotator_options}">
    <div class="rotator-wrapp">
            <h1 class="cd-headline letters type {$watcher}" data-rev-settings="{$data_rev_settings}">{$text}</h1>
    </div>
    {$scroll_arrow_html}
</div>
HTML;
        
    }
    
    protected function _content_template() {
        
    }
}
