<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Widget_Portfolios_Case_Study extends Widget_Base {
    
    public function get_name() {
        return 'litae-portfolios-case-study';
    }
    
    public function get_title() {
        return esc_html__( 'Portfolios Case Study', 'elementor-custom-element' );
    }
    
    public function get_icon() {
        return 'fa fa-briefcase';
    }
    
    public function get_categories() {
        return [ 'litae-elements' ];
    }
    
    public function get_script_depends() {
        return [ 'jquery' ];
    }
    
    protected function _register_controls() {
        $this->start_controls_section(
            'section_case_study',
            [
                'label' => esc_html__( 'Portfolios', 'elementor-custom-element' ),
            ]
        );
        $this->add_control(
            'item_height',
            [
                'label' => esc_html__( 'Section height', 'elementor-custom-element' ),
                'type' => Controls_Manager::SELECT,
                'default' => '100vh',
                'options' => [
                    '50vh'  => esc_html__( '50vh', 'elementor-custom-element' ),
                    '80vh'  => esc_html__( '80vh', 'elementor-custom-element' ),
                    '100vh' => esc_html__( '100vh', 'elementor-custom-element' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .portfolios-case-study' => 'height: {{VALUE}};',
                ],
            ]
        );
        /**
        * Repeater
        */
       $this->add_control(
           'case_study',
           [
               'label' => esc_html__( 'Items', 'elementor-custom-element' ),
               'type' => Controls_Manager::REPEATER,
               'default' => [],
               'title' => 'Choose portfolio',
               'fields' => [
                   [
                       'name' => 'portfolio',
                       'label' => esc_html__( 'Portfolio', 'elementor-custom-element' ),
                       'label_block' => true,
                       'type' => Controls_Manager::SELECT2,
                       'options' => $this->getAllPortfolios(),
                       'multiple' => false,
                   ],
                   [
                        'name' => 'is_effect',
                        'label' => esc_html( 'Reveal effect', 'elementor-custom-element' ),
                        'type' => Controls_Manager::SWITCHER,
                        'default' => false,
                   ],
                   [
                        'name' => 'animation_color',
                        'label' => __( 'Animation Color', 'elementor-custom-element' ),
                        'type' => Controls_Manager::COLOR,
                        'scheme' => [
                            'type' => Scheme_Color::get_type(),
                            'value' => Scheme_Color::COLOR_1,
                        ],
                        'condition' => [
                            'is_effect' => 'on'
                        ]
                   ],
                   [
                       'name' => 'animation_direction',
                       'label' => esc_html__( 'Animation direction', 'elementor-custom-element' ),
                       'type' => Controls_Manager::SELECT,
                       'label_block' => true,
                       'options' => [
                          'lr' => esc_html__( 'Left to right', 'elementor-custom-element' ),
                          'rl' => esc_html__( 'Right to left', 'elementor-custom-element' ),
                          'tb' => esc_html__( 'Top to bottom', 'elementor-custom-element' ),
                          'bt' => esc_html__( 'Bottom to top', 'elementor-custom-element' ),
                       ],
                       'default' => 'lr',
                       'multiple' => false,
                       'condition' => [
                           'is_effect' => 'on'
                       ]
                    ],
                    [
                        'name' => 'animation_delay',
                        'label' => __( 'Animation delay', 'elementor-custom-element' ),
                        'type' => Controls_Manager::SLIDER,
                        'default' => [
                            'size' => 250,
                        ],
                        'range' => [
                            'px' => [
                                'min' => 0,
                                'max' => 2000,
                                'step' => 10,
                            ],
                        ],
                        'size_units' => [ 'px' ],
                        'condition' => [
                            'is_effect' => 'on'
                        ]
                    ]
               ],
               //'title_field' => '{{{ text }}}',
           ]
       );
        $this->end_controls_section();
    }
    
    protected function render() {
        $settings = $this->get_settings();
        
        if( empty( $settings['case_study'] ) ){
            return;
        }
        
        $case_study_name = '';
        $case_study_images = '';
        foreach ( $settings['case_study'] as $item ):
            $id = (int)$item['portfolio'];
            $terms = wp_get_object_terms( $id, 'portfolio_tag' );
            $tags_list_name = '';
            if( ! empty($terms) && ! is_wp_error($terms) ){
                $tags = $this->getTagsArray($terms);
                $tags_list_name  .= esc_html( $this->getTagsList($tags,'name') );
            }
            
            $is_effect = ($item['is_effect']!='on')? false : true;
            $watcher = $is_effect? 'watcher' : '';
            $animation_settings = [
                'bgcolor'     => $item['animation_color'],
                'direction' => $item['animation_direction'],
                'delay'     => absint( $item['animation_delay']['size'] ),
            ];
            $data_rev_settings = esc_attr( wp_json_encode( $animation_settings ) );
            
            $case_study_name .= '<li class="case-study-name '.$watcher.'" data-rev-settings="'.$data_rev_settings.'">';
            $case_study_name .= '<a href="'.get_post_permalink($id).'">' . get_the_title($id);
            $case_study_name .= ( '' != $tags_list_name)? '<br><span>'.$tags_list_name.'</span>'  : '';
            $case_study_name .= '</a>';
            $case_study_name .= '</li>';
            if (has_post_thumbnail($id)) { // check if the post has a Post Thumbnail assigned to it.
                $case_study_images .= '<li style="background-image: url(\''. get_the_post_thumbnail_url( $id, 'full' ) .'\')">';
                $case_study_images .= '</li>';
            }
        endforeach;
        
        echo <<<HTML
<div class="portfolios-case-study">
    <ul class="case-study-wrapper">
        {$case_study_name}
    </ul>
    <div class="light-over-portfolio"></div>	
    <ul class="case-study-images">
        {$case_study_images}
    </ul>
</div>
HTML;
    }
    
    protected function _content_template() {
        
    }
    
    public function getAllPortfolios() {
        $return = [];
        $loop = new \WP_Query( array( 'post_type' => 'portfolio', 'posts_per_page' => -1, 'post_status' => 'publish', 'orderby' => 'post_date', 'order' => 'desc' ) );
        while ( $loop->have_posts() ) : $loop->the_post();
            $title = get_the_title();
            $return[get_the_ID()] = $title;
        endwhile;
        wp_reset_postdata();
        wp_reset_query();
        
        return $return;
    }
    
    protected function getTagsList($tags, $selector='name', $separator=' ') {
        $return = '';
        foreach( $tags as $tag ):
            $return .= $tag[$selector] . $separator;
        endforeach;
        
        return $return;
    }
    
    protected function getTagsArray($terms) {
        $return = array();
        foreach( $terms as $term ):
            $return[] = [
                'slug'  => $term->slug,
                'name'  => $term->name,
                'count' => $term->count
            ];
        endforeach;
        
        return $return;
    }
}
