<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Piroll
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
if ( ! function_exists( 'piroll_body_classes' ) ) :
    function piroll_body_classes( $classes ) {
        // Adds a class of group-blog to blogs with more than 1 published author.
        if ( is_multi_author() ) {
            $classes[] = 'group-blog';
        }

        // Adds a class of hfeed to non-singular pages.
        if ( ! is_singular() ) {
            $classes[] = 'hfeed';
        }

        return $classes;
    }
endif;
add_filter( 'body_class', 'piroll_body_classes' );


// return current page post type. If post type NOT blog - return page
if ( ! function_exists( 'piroll_post_type_is' ) ) :
    function piroll_post_type_is() {
        $result = 'page';

        if ( is_search() ) {
            $result = 'search';
        } else if ( is_404() ) {
            $result = '404';
        } else if ( is_home() || is_archive() || is_category() || is_tag() || is_date() ) {
            if ( get_post_type() == 'portfolio' ) {
                $result = 'portfolio';
            } else {
                $result = 'archive';
            }
        }

        return $result;
    }
endif;

// print nav logo
if ( ! function_exists( 'piroll_nav_logo' ) ) :
    function piroll_nav_logo() {
        $post_type = get_post_type();
        if ( ! $post_type ) {
            $post_type = 'page';
        }

        $acf_navigation = piroll_get_theme_mod( $post_type . '_navigation_custom', true );

        $logo = piroll_get_attachment( piroll_get_theme_mod( 'navigation_logo' ) );
        $logo_light = piroll_get_attachment( piroll_get_theme_mod( 'navigation_logo_light' ) );

        ob_start();
        ?>

        <?php if ( $logo ) : ?>
            <a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="nk-nav-logo">
                <?php if ( piroll_get_theme_mod( 'navigation_white_top_text', $acf_navigation ) ) : ?>
                    <?php
                    $alt = $logo_light['alt'];
                    if ( ! $logo_light['alt'] ) {
                        $alt = get_bloginfo( 'name' );
                    }
                    ?>
                    <img src="<?php echo esc_url( $logo_light['src'] ); ?>" alt="<?php echo esc_attr( $alt ); ?>" width="<?php echo esc_attr( piroll_get_theme_mod( 'navigation_logo_width' ) ); ?>" class="nk-nav-logo-onscroll">
                    <?php
                endif;

                $alt = $logo['alt'];
                if ( ! $logo['alt'] ) {
                    $alt = get_bloginfo( 'name' );
                }
                ?>
                <img src="<?php echo esc_url( $logo['src'] ); ?>" alt="<?php echo esc_attr( $alt ); ?>" width="<?php echo esc_attr( piroll_get_theme_mod( 'navigation_logo_width' ) ); ?>">
            </a>
            <?php
        endif;
        $return = ob_get_contents();
        ob_end_clean();

        return $return;
    }
endif;

/**
 * Get Attachment Attribute for Images
 */
if ( ! function_exists( 'piroll_get_attachment' ) ) :
    function piroll_get_attachment( $attachment_id, $attachment_size = 'full' ) {
        // is url
        if ( filter_var( $attachment_id, FILTER_VALIDATE_URL ) ) {
            $path_to_image = $attachment_id;
            $attachment_id = attachment_url_to_postid( $attachment_id );
            if ( is_numeric( $attachment_id ) && $attachment_id == 0 ) {
                return array(
                    'alt' => null,
                    'caption' => null,
                    'description' => null,
                    'href' => null,
                    'src' => $path_to_image,
                    'title' => null,
                    'width' => null,
                    'height' => null,
                );
            }
        }

        // is numeric
        if ( is_numeric( $attachment_id ) && $attachment_id !== 0 ) {
            $attachment = get_post( $attachment_id );
            if ( is_object( $attachment ) ) {
                $attachment_src = array();
                if ( isset( $attachment_size ) ) {
                    $attachment_src = wp_get_attachment_image_src( $attachment_id, $attachment_size );
                }
                return array(
                    'alt' => get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true ),
                    'caption' => $attachment->post_excerpt,
                    'description' => $attachment->post_content,
                    'href' => get_permalink( $attachment->ID ),
                    'src' => isset( $attachment_src[0] ) ? $attachment_src[0] : $attachment->guid,
                    'title' => $attachment->post_title,
                    'width' => isset( $attachment_src[1] ) ? $attachment_src[1] : false,
                    'height' => isset( $attachment_src[2] ) ? $attachment_src[2] : false,
                );
            }
        }
        return false;
    }
endif;
/**
 * Get Attachment Attribute for Images
 */
if ( ! function_exists( 'piroll_get_social_list' ) ) :
    function piroll_get_social_list( $size_list = 'full' ) {
        if ( $size_list == 'full' ) {
            return array(
                'fa fa-twitch' => esc_html__( 'Twitch', 'piroll' ),
                'fa fa-vimeo' => esc_html__( 'Vimeo', 'piroll' ),
                'fa fa-youtube' => esc_html__( 'Youtube', 'piroll' ),
                'fa fa-facebook' => esc_html__( 'Facebook', 'piroll' ),
                'fa fa-google-plus' => esc_html__( 'Google-plus', 'piroll' ),
                'fa fa-twitter' => esc_html__( 'Twitter', 'piroll' ),
                'fa fa-pinterest' => esc_html__( 'Pinterest', 'piroll' ),
                'fa fa-instagram' => esc_html__( 'Instagram', 'piroll' ),
                'fa fa-linkedin' => esc_html__( 'Linkedin', 'piroll' ),
                'fa fa-behance' => esc_html__( 'Behance', 'piroll' ),
                'fa fa-odnoklassniki' => esc_html__( 'Odnoklassniki', 'piroll' ),
                'fa fa-skype' => esc_html__( 'Skype', 'piroll' ),
                'fa fa-vk' => esc_html__( 'VK', 'piroll' ),
                'fa fa-steam' => esc_html__( 'Steam', 'piroll' ),
                'fa fa-bitbucket' => esc_html__( 'Bitbucket', 'piroll' ),
                'fa fa-dropbox' => esc_html__( 'Dropbox', 'piroll' ),
                'fa fa-dribbble' => esc_html__( 'Dribbble', 'piroll' ),
                'fa fa-deviantart' => esc_html__( 'Deviantart', 'piroll' ),
                'fa fa-flickr' => esc_html__( 'Flickr', 'piroll' ),
                'fa fa-foursquare' => esc_html__( 'Foursquare', 'piroll' ),
                'fa fa-github' => esc_html__( 'Github', 'piroll' ),
                'fa fa-medium' => esc_html__( 'Medium', 'piroll' ),
                'fa fa-paypal' => esc_html__( 'PayPal', 'piroll' ),
                'fa fa-reddit' => esc_html__( 'Reddit', 'piroll' ),
                'fa fa-soundcloud' => esc_html__( 'SoundCloud', 'piroll' ),
                'fa fa-slack' => esc_html__( 'Slack', 'piroll' ),
                'fa fa-tumblr' => esc_html__( 'Tumblr', 'piroll' ),
                'fa fa-wordpress' => esc_html__( 'WordPress', 'piroll' ),
            );
        } elseif ( $size_list == 'small' ) {
            return array(
                'fa fa-facebook' => esc_html__( 'Facebook', 'piroll' ),
                'fa fa-google-plus' => esc_html__( 'Google Plus', 'piroll' ),
                'fa fa-twitter' => esc_html__( 'Twitter', 'piroll' ),
                'fa fa-pinterest' => esc_html__( 'Pinterest', 'piroll' ),
                'fa fa-linkedin' => esc_html__( 'Linkedin', 'piroll' ),
                'fa fa-vk' => esc_html__( 'VK', 'piroll' ),
            );
        } else {
            return null;
        }
    }
endif;

/**
 * Get all terms for post shortcodes
 */
if ( ! function_exists( 'piroll_get_terms' ) ) :
    function piroll_get_terms() {
        $terms_list_vc = array();
        $terms_list = get_terms(
            get_object_taxonomies(
                get_post_types(
                    array(
                        'public' => false,
                        'name' => 'attachment',
                    ), 'names', 'NOT'
                )
            )
        );
        foreach ( $terms_list as $term ) {
            $terms_list_vc[] = array(
                'value' => $term->term_id,
                'label' => $term->name,
                'group' => $term->taxonomy,
            );
        }

        return $terms_list_vc;
    }
endif;



/**
 * Responsive video embed
 */
add_filter( 'embed_oembed_html', 'piroll_oembed_filter', 10, 4 );
if ( ! function_exists( 'piroll_oembed_filter' ) ) :
    function piroll_oembed_filter( $html, $url, $attr, $post_ID ) {
        $classes = '';
        if ( strpos( $url, 'youtube' ) > 0 || strpos( $url, 'youtu.be' ) > 0 ) {
            $classes .= ' responsive-embed responsive-embed-16x9 embed-youtube';
        } else if ( strpos( $url, 'vimeo' ) > 0 ) {
            $classes .= ' responsive-embed responsive-embed-16x9 embed-vimeo';
        } else if ( strpos( $url, 'twitter' ) > 0 ) {
            $classes .= ' embed-twitter';
        }

        $return = '<div class="' . esc_attr( $classes ) . '">' . $html . '</div>';
        return $return;
    }
endif;

// print filter block in blog/works list
if ( ! function_exists( 'piroll_get_isotope_filter' ) ) :
    function piroll_get_isotope_filter( $opened = false, $custom_query = false ) {
        $result = '<div class="nk-gap-4"></div>';

        if ( ! $custom_query ) {
            global $wp_query;
            $custom_query = $wp_query;
        }

        if ( ! is_category() && ! is_tax( 'portfolio-category' ) ) {
            $categories = array();

            // get all categories from all available posts
            if ( $custom_query && $custom_query->have_posts() ) {
                while ( $custom_query->have_posts() ) :
                    $custom_query->the_post();
                    $item_categories = get_the_terms( get_the_ID(), 'portfolio-category' );
                    if ( ! $item_categories || ! is_array( $item_categories ) ) {
                        $item_categories = get_the_terms( get_the_ID(), 'category' );
                    }
                    if ( $item_categories && is_array( $item_categories ) ) {
                        foreach ( $item_categories as $item ) {
                            $categories[] = array(
                                'slug' => $item->slug,
                                'name' => $item->name,
                                'term_id' => $item->term_id,
                            );
                        }
                    }
                endwhile;
            }
            $categories = array_unique( $categories, SORT_REGULAR );

            $categories_string = '';
            foreach ( $categories as $cat ) {
                $categories_string .= '<li data-filter="' . esc_attr( $cat['slug'] ) . '" data-term-id="' . esc_attr( $cat['term_id'] ) . '">' . $cat['name'] . '</li> ';
            }
            $result = '
                <div class="nk-pagination nk-pagination-nobg nk-pagination-center' . ( $opened ? ' nk-isotope-filter-active' : '' ) . '">
                    <a href="#nk-toggle-filter"><span class="nk-icon-squares"></span></a>
                </div>
                <ul class="nk-isotope-filter' . ( $opened ? ' nk-isotope-filter-active' : '' ) . '">
                    <li class="active" data-filter="*">' . esc_html__( 'All', 'piroll' ) . '</li>
                    ' . $categories_string . '
                </ul>
            ';
        }
        return $result;
    }
endif;

// get portfolio classes
if ( ! function_exists( 'piroll_get_portfolio_classes' ) ) :
    function piroll_get_portfolio_classes( $list_style = 'masonry_3_style_1', $items_gap = '', $items_hover_style = '1_left_bottom', $item_id = 1 ) {
        $parent_class = 'nk-portfolio-list nk-isotope';
        $item_class = 'nk-portfolio-item';
        $items_classes = array( '' );
        $item_info_class = '';
        $isotope_item_class = 'nk-isotope-item';
        $isotope_items_classes = array( '' );
        $image_size = '';
        $images_sizes = array( 'piroll_800x600' );

        // add list style
        switch ( $list_style ) {
            case 'masonry_1_style_1':
                $parent_class .= ' nk-isotope-1-cols';
                $items_classes = array(
                    'nk-portfolio-item-xxs',
                );
                $images_sizes = array(
                    'piroll_1920x1080',
                );
                break;
            case 'masonry_2_style_1':
                $parent_class .= ' nk-isotope-2-cols';
                $images_sizes = array(
                    'piroll_1280x720',
                );
                break;
            case 'masonry_2_style_2':
                $parent_class .= ' nk-isotope-2-cols';
                $items_classes = array(
                    'nk-portfolio-item-square',
                );
                $images_sizes = array(
                    'piroll_1280x720',
                );
                break;
            case 'masonry_2_style_3':
                $parent_class .= ' nk-isotope-2-cols';
                $items_classes = array(
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-sm',
                );
                $images_sizes = array(
                    'piroll_1280x720',
                    'piroll_1920x1080',
                    'piroll_1920x1080',
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_1280x720',
                );
                break;
            case 'masonry_3_style_2':
                $parent_class .= ' nk-isotope-3-cols';
                $items_classes = array(
                    'nk-portfolio-item-square',
                );
                break;
            case 'masonry_3_style_3':
                $parent_class .= ' nk-isotope-3-cols';
                $items_classes = array(
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-sm',
                    'nk-portfolio-item-sm',
                );
                $images_sizes = array(
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                );
                if ( $items_gap !== 'mid' && $items_gap !== 'big' ) {
                    $items_classes = str_replace( 'nk-portfolio-item-lg', 'nk-portfolio-item-lg-0', $items_classes );
                }
                break;
            case 'masonry_3_style_4':
                $parent_class .= ' nk-isotope-3-cols';
                $items_classes = array(
                    'nk-portfolio-item-xl',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-xs',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                );
                $isotope_items_classes = array(
                    '',
                    '',
                    '',
                    '',
                    '',
                    '',
                    'nk-isotope-item-x2',
                    '',
                    '',
                    '',
                );
                $images_sizes = array(
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_800x600',
                );
                break;
            case 'masonry_3_metro_1':
                $parent_class .= ' nk-isotope-3-cols';
                $isotope_items_classes = array(
                    '',
                    'nk-isotope-item-x2',
                    '',
                    '',
                    '',
                    'nk-isotope-item-x2',
                    '',
                    '',
                );
                $items_classes = array(
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-xs',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                );
                $images_sizes = array(
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                );
                break;
            case 'masonry_3_metro_2':
                $parent_class .= ' nk-isotope-3-cols';
                $isotope_items_classes = array(
                    '',
                    'nk-isotope-item-x2',
                    '',
                    '',
                    'nk-isotope-item-x2',
                );
                $items_classes = array(
                    'nk-portfolio-item-xl',
                    'nk-portfolio-item-xs',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-xl',
                    'nk-portfolio-item-xs',
                );
                $images_sizes = array(
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_1280x720',
                );
                break;
            case 'masonry_4_style_1':
                $parent_class .= ' nk-isotope-4-cols';
                $items_classes = array(
                    'nk-portfolio-item-square',
                );
                break;
            case 'masonry_4_style_2':
                $parent_class .= ' nk-isotope-4-cols';
                $items_classes = array(
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-lg',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-square',
                );
                break;
            case 'masonry_4_metro_1':
                $parent_class .= ' nk-isotope-4-cols';
                $isotope_items_classes = array(
                    '',
                    '',
                    'nk-isotope-item-x2',
                    '',
                    '',
                    'nk-isotope-item-x2',
                    '',
                    '',
                    '',
                    '',
                );
                $items_classes = array(
                    'nk-portfolio-item-square',
                );
                $images_sizes = array(
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_1920x1080',
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_1920x1080',
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_1280x720',
                    'piroll_1280x720',
                );
                break;
            case 'masonry_4_metro_2':
                $parent_class .= ' nk-isotope-4-cols';
                $isotope_items_classes = array(
                    'nk-isotope-item-x2',
                );
                $items_classes = array(
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-xs',
                    'nk-portfolio-item-xs',
                    'nk-portfolio-item-xs',
                    'nk-portfolio-item-square',
                    'nk-portfolio-item-xs',
                );
                $images_sizes = array(
                    'piroll_1280x720',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_800x600',
                    'piroll_1280x720',
                    'piroll_800x600',
                );
                break;
            default:
                $parent_class .= ' nk-isotope-3-cols';
        }

        // add gap
        switch ( $items_gap ) {
            case 'small':
                $parent_class .= ' nk-isotope-gap-small';
                break;
            case 'mid':
                $parent_class .= ' nk-isotope-gap';
                break;
            case 'big':
                $parent_class .= ' nk-isotope-gap-big';
                break;
        }

        // add hover style
        switch ( $items_hover_style ) {
            case '1_left_bottom':
            case '2_left_bottom':
                $item_info_class = 'nk-portfolio-item-info-bottom text-xs-left';
                break;
            case '1_right_bottom':
            case '2_right_bottom':
                $item_info_class = 'nk-portfolio-item-info-bottom text-xs-right';
                break;
            case '1_right_top':
            case '2_right_top':
                $item_info_class = 'nk-portfolio-item-info-top text-xs-right';
                break;
            case '1_left_top':
            case '2_left_top':
                $item_info_class = 'nk-portfolio-item-info-top text-xs-left';
                break;
            case '1_center':
            case '2_center':
                $item_info_class = 'nk-portfolio-item-info-center text-xs-center';
                break;
            case '1_left':
            case '2_left':
                $item_info_class = 'nk-portfolio-item-info-center text-xs-left';
                break;
            case '1_right':
            case '2_right':
                $item_info_class = 'nk-portfolio-item-info-center text-xs-right';
                break;
            case '1_top':
            case '2_top':
                $item_info_class = 'nk-portfolio-item-info-top text-xs-center';
                break;
            case '1_bottom':
            case '2_bottom':
                $item_info_class = 'nk-portfolio-item-info-bottom text-xs-center';
                break;
            case '3_left':
                $item_info_class = 'text-xs-left';
                break;
            case '3_center':
                $item_info_class = 'text-xs-center';
                break;
            case '3_right':
                $item_info_class = 'text-xs-right';
                break;
        }
        if ( strpos( $items_hover_style, '2_' ) !== false ) {
            $item_class .= ' hover';
        }
        if ( strpos( $items_hover_style, '3_' ) !== false ) {
            $item_class .= ' nk-portfolio-item-info-style-3';
        } else if ( $items_hover_style == 'none' ) {
            $item_class .= ' nk-portfolio-item-info-style-4';
        } else {
            $item_class .= ' nk-portfolio-item-info-style-1';
        }

        if ( count( $isotope_items_classes ) > 1 ) {
            $isotope_item_id = $item_id % count( $isotope_items_classes );
            $isotope_item_class .= ' ' . $isotope_items_classes[ $isotope_item_id ];
        } else {
            $isotope_item_class .= ' ' . $isotope_items_classes[0];
        }

        if ( count( $items_classes ) > 1 ) {
            $classes_item_id = $item_id % count( $items_classes );
            $item_class .= ' ' . $items_classes[ $classes_item_id ];
        } else {
            $item_class .= ' ' . $items_classes[0];
        }

        if ( count( $images_sizes ) > 1 ) {
            $img_sizes_item_id = $item_id % count( $images_sizes );
            $image_size = $images_sizes[ $img_sizes_item_id ];
        } else {
            $image_size = $images_sizes[0];
        }

        return array(
            'parent_class' => $parent_class,
            'isotope_item_class' => $isotope_item_class,
            'item_class' => $item_class,
            'item_info_class' => $item_info_class,
            'image_size' => $image_size,
        );
    }
endif;


if ( ! function_exists( 'piroll_password_form' ) ) :
    function piroll_password_form() {
        global $post;
        $label = 'pwbox-' . ( empty( $post->ID ) ? rand() : $post->ID );
        $o = '<form action="' . esc_url( site_url( 'wp-login.php?action=postpass', 'login_post' ) ) . '" method="post" class="nk-form nk-form-style-1 validate"><div class="input-group"><input name="post_password" id="' . $label . '" type="password" class="form-control required" placeholder = "' . esc_html__( 'Password', 'piroll' ) . '"/><span class="nk-input-group-btn"><button class="nk-btn password-submit">' . esc_html__( 'Submit', 'piroll' ) . '</button></span></div><small>' . esc_html__( 'To view this protected post, enter the password below', 'piroll' ) . '</small> </form>
        ';
        return $o;
    }
endif;
add_filter( 'the_password_form', 'piroll_password_form' );
